<?php

class PixPayment
{
    private $apiUrl;
    private $apiToken;

    public function __construct($apiUrl, $apiToken)
    {
        $this->apiUrl = $apiUrl;
        $this->apiToken = $apiToken;
    }
    
    private function generateNumericOrderId()
    {
        return time() . mt_rand(1000, 9999);
    }

    // Gera UUID (para orderId e productId)
    private function generateUUID()
    {
        return sprintf('%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
            mt_rand(0, 0xffff), mt_rand(0, 0xffff),
            mt_rand(0, 0xffff),
            mt_rand(0, 0x0fff) | 0x4000,
            mt_rand(0, 0x3fff) | 0x8000,
            mt_rand(0, 0xffff), mt_rand(0, 0xffff), mt_rand(0, 0xffff)
        );
    }

    // Método para registrar erro em log
    private function logError($message, $data = [])
    {
        // Cria o nome do arquivo de log com data e hora
        $logFileName = 'LOG-' . date('Y-m-d_H-i-s') . '_ERRO_ENVIANDO_ITEM_PAGO.txt';
        
        // Formata a mensagem de erro
        $logContent = "[" . date('Y-m-d H:i:s') . "] " . $message . "\n";
        
        // Adiciona dados extras se houver
        if (!empty($data)) {
            $logContent .= "Dados: " . json_encode($data, JSON_PRETTY_PRINT) . "\n";
        }
        
        // Tenta escrever no arquivo
        try {
            file_put_contents($logFileName, $logContent, FILE_APPEND);
            return true;
        } catch (Exception $e) {
            error_log("Erro ao escrever no arquivo de log: " . $e->getMessage());
            return false;
        }
    }

    // Função para enviar a requisição à API
    public function sendRequest($data)
    {
        $headers = [
            "Content-Type: application/json",
            "x-api-token: " . $this->apiToken
        ];

        $ch = curl_init($this->apiUrl);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        return ['response' => $response, 'status_code' => $httpCode];
    }

    // Função para converter preço float para centavos (inteiro)
    private function convertToCents($price)
    {
        try {
            // Converte para string para garantir precisão
            $priceStr = (string)$price;
            
            // Verifica se o preço é válido
            if (empty($priceStr) || !is_numeric(str_replace(',', '.', $priceStr))) {
                $this->logError("Preço inválido para conversão", ['preco' => $priceStr]);
                return false; // Retorna falso ao invés de um valor padrão
            }
            
            // Remove possíveis formatações e usa . como separador decimal
            $priceStr = str_replace(',', '.', $priceStr);
            
            // Multiplica por 100 e arredonda para evitar problemas com floats
            $cents = (int)round(floatval($priceStr) * 100);
            
            // Garantia de valor mínimo para evitar problemas
            return max($cents, 1); // Mínimo de 1 centavo
        } catch (Exception $e) {
            $this->logError("Erro ao converter preço para centavos", [
                'preco' => $price, 
                'erro' => $e->getMessage()
            ]);
            return false; // Retorna falso ao invés de um valor padrão
        }
    }

    // Nova função para processar o pedido de Pix com produtos do carrinho
    public function sendPix($customerData, $status)
    {
        $responses = [];
        
        // FALLBACK 1: Verifica se existe carrinho e se tem itens
        if (isset($customerData['cart']) && isset($customerData['cart']['itens']) && !empty($customerData['cart']['itens'])) {
            $cartItems = $customerData['cart']['itens'];
            
            // Verificação de segurança para validar os dados de cada item
            $validItems = [];
            foreach ($cartItems as $item) {
                if (isset($item['nome']) && isset($item['preco'])) {
                    $validItems[] = $item;
                }
            }
            
            // FALLBACK 2: Se tem pelo menos 1 item válido, processa os itens
            if (!empty($validItems)) {
                // Para cada item do carrinho, cria uma requisição
                foreach ($validItems as $item) {
                    try {
                        $priceInCents = $this->convertToCents($item['preco']);
                        
                        // Se a conversão falhou, pula este item
                        if ($priceInCents === false) {
                            $this->logError("Item ignorado devido a preço inválido", $item);
                            continue;
                        }
                        
                        // Obtém a quantidade do produto (mínimo 1)
                        $quantity = (isset($item['quantidade']) && $item['quantidade'] > 0) ? $item['quantidade'] : 1;
                        
                        // Calcula o preço total considerando a quantidade
                        $totalPriceInCents = $priceInCents * $quantity;
                        
                        // Calcula taxa e comissão com base no preço total
                            //$gatewayFeeInCents = (int)round($totalPriceInCents * 0.066); // 6.6% de taxa
                            //$userCommissionInCents = $totalPriceInCents - $gatewayFeeInCents;

                        //Como na UTM já definimos as taxas, passar a taxa do gate = 0 / Valor final pos txs gate = valor ticket
                        $gatewayFeeInCents = 0;
                        $userCommissionInCents = $totalPriceInCents;

                        $mockedData = [
                            'orderId' => $this->generateNumericOrderId(),
                            'platform' => 'AtivoPay',
                            'paymentMethod' => 'pix',
                            'status' => $status,
                            'createdAt' => date('Y-m-d H:i:s'),
                            'approvedDate' => date('Y-m-d H:i:s'),
                            'refundedAt' => null,
                            'customer' => [
                                'name' => $customerData['name'],
                                'email' => $customerData['email'],
                                'phone' => $customerData['phone'],
                                'document' => $customerData['document'],
                                'country' => 'BR'
                            ],
                            'products' => [
                                [
                                    'id' => $item['id'] ?? $this->generateNumericOrderId(),
                                    'name' => $item['nome'],
                                    'planId' => null,
                                    'planName' => null,
                                    'quantity' => $quantity,
                                    'priceInCents' => $priceInCents
                                ]
                            ],
                            'trackingParameters' => [
                                'utm_source' => $customerData['trackingParameters']['utm_source'] ?? null,
                                'utm_campaign' => $customerData['trackingParameters']['utm_campaign'] ?? null,
                                'utm_medium' => $customerData['trackingParameters']['utm_medium'] ?? null,
                                'utm_content' => $customerData['trackingParameters']['utm_content'] ?? null,
                                'utm_term' => $customerData['trackingParameters']['utm_term'] ?? null
                            ],
                            'commission' => [
                                'totalPriceInCents' => $totalPriceInCents,
                                'gatewayFeeInCents' => $gatewayFeeInCents,
                                'userCommissionInCents' => $userCommissionInCents
                            ],
                            'isTest' => false
                        ];

                        $responses[] = $this->sendRequest($mockedData);
                    } catch (Exception $e) {
                        // Se houver exceção ao processar um item, registra o erro e continua
                        $this->logError("Erro ao processar item", [
                            'item' => $item,
                            'erro' => $e->getMessage()
                        ]);
                        continue;
                    }
                }
                
                // Se pelo menos um produto foi processado com sucesso
                if (!empty($responses)) {
                    return [
                        'response' => json_encode(['results' => $responses]),
                        'status_code' => 200
                    ];
                }
            }
        }
        
        // Se nenhuma condição anterior foi atendida, registra erro e retorna resposta de erro
        $errorMessage = "Erro ao processar pedido: Carrinho inválido ou vazio";
        $this->logError($errorMessage, ['customerData' => $customerData]);
        
        return [
            'response' => json_encode([
                'error' => $errorMessage,
                'timestamp' => date('Y-m-d H:i:s')
            ]),
            'status_code' => 400
        ];
    }

}