// Sistema de Validação de Checkout com Feedback Personalizado
document.addEventListener("DOMContentLoaded", function () {
  // Cache dos elementos principais
  const formIdentificacao = document.getElementById("cld-identificacao");
  const formEntrega = document.getElementById("entrega-form");
  const formPagamento = document.getElementById("pagamento-form");

  // Botões de navegação
  const btnParaEntrega = document.getElementById("cld_delivery_next_step");
  const btnParaPagamento = document.getElementById("cld_payment_next_step");
  const btnVoltarParaIdentificacao =
    document.getElementById("cld_back_contact");
  const btnVoltarParaEntrega = document.getElementById("cld_back_delivery");
  const btnFinalizarPix = document.getElementById("cld_finalize_pix_purchase");
  const btnFinalizarCartao = document.getElementById(
    "cld_finalize_credit_card_purchase"
  );

  // Métodos de pagamento
  const pixPayment = document.getElementById("pix-payment-0");
  const cardPayment = document.getElementById("card-payment-0");
  const pixDataPayment = document.getElementById("pix_data_payment");
  const cardDataPayment = document.getElementById("card_data_payment");

  console.log("Script inicializado"); // Debug

  // Função para validar e-mail
  function validarEmail(email) {
    const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return re.test(email);
  }

  // Função para validar CPF
  function validarCPF(cpf) {
    cpf = cpf.replace(/[^\d]/g, "");

    if (cpf.length !== 11) return false;

    // Verificação simplificada para fins de demonstração
    return cpf.length === 11;
  }

  // Função para validar telefone
  function validarTelefone(telefone) {
    telefone = telefone.replace(/[^\d]/g, "");
    return telefone.length >= 10 && telefone.length <= 11;
  }

  // Função para validar CEP
  function validarCEP(cep) {
    cep = cep.replace(/[^\d]/g, "");
    return cep.length === 8;
  }

  // Função para mostrar feedback de erro
  function mostrarErro(campo, mensagem) {
    console.log("Mostrando erro para campo", campo.id, mensagem); // Debug

    // Remover mensagem de erro anterior se existir
    const erroAnterior = document.getElementById(`cld-error-${campo.id}`);
    if (erroAnterior) {
      erroAnterior.remove();
    }

    // Criar elemento de feedback personalizado
    const feedbackDiv = document.createElement("div");
    feedbackDiv.id = `cld-error-${campo.id}`;
    feedbackDiv.className = "cld-invalid-feedback";
    feedbackDiv.style.color = "#ff0000";
    feedbackDiv.style.fontSize = "12px";
    feedbackDiv.style.marginTop = "5px";
    feedbackDiv.style.display = "block";

    // Criar ícone de alerta
    const imgAlert = document.createElement("img");
    imgAlert.src = "assets/img/error_cupom.svg";
    imgAlert.alt = "";
    imgAlert.className = "alert_cupom";
    imgAlert.style.marginRight = "5px";
    imgAlert.style.display = "inline-block";

    // Criar span para a mensagem
    const mensagemSpan = document.createElement("span");
    mensagemSpan.className = "cld-invalid-data";
    mensagemSpan.textContent = mensagem;
    mensagemSpan.style.fontSize = "12px";

    // Montar a estrutura
    feedbackDiv.appendChild(imgAlert);
    feedbackDiv.appendChild(mensagemSpan);

    // Adicionar classe de erro ao campo
    campo.classList.add("cld-is-invalid");
    campo.style.borderColor = "#ff0000";

    // Inserir o feedback após o campo
    campo.parentNode.insertBefore(feedbackDiv, campo.nextSibling);
  }

  // Função para limpar feedback de erro
  function limparErro(campo) {
    const erroDiv = document.getElementById(`cld-error-${campo.id}`);
    if (erroDiv) {
      erroDiv.remove();
    }
    campo.classList.remove("cld-is-invalid");
    campo.style.borderColor = "";
  }

  // Função para validar o formulário de identificação
  function validarFormIdentificacao() {
    console.log("Validando formulário de identificação"); // Debug
    let valido = true;

    // Validar e-mail
    const email = document.getElementById("cld_email");
    const noEmail = document.getElementById("cld_noEmail");

    if (!noEmail.checked && (!email.value || !validarEmail(email.value))) {
      mostrarErro(email, "E-mail inválido. Ex: email@email.com");
      valido = false;
    } else {
      limparErro(email);
    }

    // Validar telefone
    const telefone = document.getElementById("cld_telephone");
    if (!telefone.value || !validarTelefone(telefone.value)) {
      mostrarErro(telefone, "Telefone inválido. Ex: (51) 99610-4598");
      valido = false;
    } else {
      limparErro(telefone);
    }

    // Validar nome
    const nome = document.getElementById("cld_name");
    if (!nome.value || nome.value.length < 5 || !nome.value.includes(" ")) {
      mostrarErro(nome, "Nome inválido. Ex: Pedro Fernandes");
      valido = false;
    } else {
      limparErro(nome);
    }

    // Validar CPF
    const cpf = document.getElementById("cld_document");
    if (!cpf.value || !validarCPF(cpf.value)) {
      mostrarErro(cpf, "CPF inválido. Ex: 200.999.745-00");
      valido = false;
    } else {
      limparErro(cpf);
    }

    console.log("Formulário de identificação válido?", valido); // Debug
    return valido;
  }

  // Função para validar o formulário de entrega
  function validarFormEntrega() {
    console.log("Validando formulário de entrega"); // Debug
    let valido = true;

    // Validar CEP
    const cep = document.getElementById("cld_zip_code");
    if (!cep.value || !validarCEP(cep.value)) {
      mostrarErro(cep, "CEP inválido. Ex: 12345-000");
      valido = false;
    } else {
      limparErro(cep);
    }

    // Validar Rua
    const rua = document.getElementById("cld_street");
    if (!rua.value || rua.value.length < 3) {
      mostrarErro(rua, "Rua inválida. Ex: Avenida Brasil");
      valido = false;
    } else {
      limparErro(rua);
    }

    // Validar Número
    const numero = document.getElementById("cld_number");
    const semNumero = document.getElementById("cld_noNumber");

    if (!semNumero.checked && (!numero.value || numero.value === "0")) {
      mostrarErro(numero, "Número inválido. Ex: 3213");
      valido = false;
    } else {
      limparErro(numero);
    }

    // Validar Bairro
    const bairro = document.getElementById("cld_neighborhood");
    if (!bairro.value || bairro.value.length < 3) {
      mostrarErro(bairro, "Bairro inválido. Ex: Centro");
      valido = false;
    } else {
      limparErro(bairro);
    }

    // Validar Cidade
    const cidade = document.getElementById("cld_city");
    if (!cidade.value || cidade.value.length < 3) {
      mostrarErro(cidade, "Cidade inválida. Ex: Gramado");
      valido = false;
    } else {
      limparErro(cidade);
    }

    // Validar Estado
    const estado = document.getElementById("cld_state");
    if (!estado.value) {
      mostrarErro(estado, "Estado inválido. Ex: Minas Gerais");
      valido = false;
    } else {
      limparErro(estado);
    }

    console.log("Formulário de entrega válido?", valido); // Debug
    return valido;
  }

  // Função para mostrar erro no CEP
  function mostrarErroCEP() {
    const cep = document.getElementById("cld_zip_code");
    mostrarErro(cep, "CEP inválido ou não encontrado.");
  }

  // Função para preencher os campos de endereço com os dados da API
  function preencherCamposEndereco(data) {
    // Preenche o campo de rua/logradouro
    const streetField = document.getElementById("cld_street");
    if (streetField) {
      streetField.value = data.logradouro || "";
      limparErro(streetField);
    }

    // Preenche o campo de bairro
    const neighborhoodField = document.getElementById("cld_neighborhood");
    if (neighborhoodField) {
      neighborhoodField.value = data.bairro || "";
      limparErro(neighborhoodField);
    }

    // Preenche o campo de cidade
    const cityField = document.getElementById("cld_city");
    if (cityField) {
      cityField.value = data.localidade || "";
      limparErro(cityField);
    }

    // Preenche o campo de estado
    const stateField = document.getElementById("cld_state");
    if (stateField) {
      stateField.value = data.uf || "";
      limparErro(stateField);
    }

    // Esconde o spinner
    const spinner = document.getElementById("zip-code-spinner");
    if (spinner) {
      spinner.style.display = "none";
    }
  }

  // Função para limpar os campos de endereço
  function limparCamposEndereco() {
    const fields = ["cld_street", "cld_neighborhood", "cld_city", "cld_state"];

    fields.forEach((field) => {
      const element = document.getElementById(field);
      if (element) {
        element.value = "";
      }
    });

    // Esconde o spinner
    const spinner = document.getElementById("zip-code-spinner");
    if (spinner) {
      spinner.style.display = "none";
    }
  }

  // Implementação da busca de CEP
  const cepInput = document.getElementById("cld_zip_code");
  if (cepInput) {
    // Adiciona um event listener para quando o campo perder o foco
    cepInput.addEventListener("blur", function () {
      // Obtém o valor do CEP digitado, removendo caracteres não numéricos
      let cep = this.value.replace(/\D/g, "");

      // Verifica se o CEP tem 8 dígitos
      if (cep.length === 8) {
        // Exibe o spinner de carregamento
        const spinner = document.getElementById("zip-code-spinner");
        if (spinner) {
          spinner.style.display = "block";
        }

        // Inicia a consulta à API do ViaCEP
        fetch(`https://viacep.com.br/ws/${cep}/json/`)
          .then((response) => {
            // Verifica se a resposta está ok
            if (!response.ok) {
              throw new Error("Erro na consulta ao CEP");
            }
            return response.json();
          })
          .then((data) => {
            // Verifica se há erro no CEP
            if (data.erro) {
              mostrarErroCEP();
              limparCamposEndereco();
              return;
            }

            // Preenche os campos com os dados retornados
            preencherCamposEndereco(data);
          })
          .catch((error) => {
            console.error("Erro ao consultar CEP:", error);
            mostrarErroCEP();
            limparCamposEndereco();
          });
      } else if (cep.length > 0) {
        // Se o CEP foi digitado, mas não tem 8 dígitos
        mostrarErroCEP();
        limparCamposEndereco();
      }
    });
  }

  // Evento ao clicar no botão de ir para entrega
  if (btnParaEntrega) {
    console.log("Botão para entrega encontrado"); // Debug
    btnParaEntrega.addEventListener("click", function (e) {
      e.preventDefault();
      console.log("Botão para entrega clicado"); // Debug

      if (validarFormIdentificacao()) {
        // Esconde formulário de identificação e mostra formulário de entrega
        formIdentificacao.style.display = "none";
        formEntrega.style.display = "block";
        formPagamento.style.display = "none";

        // Scroll para o topo do formulário
        window.scrollTo({ top: 0, behavior: "smooth" });
        console.log("Navegando para formulário de entrega"); // Debug
      }
    });
  } else {
    console.log("Botão para entrega NÃO encontrado"); // Debug
  }

  // Evento ao clicar no botão de ir para pagamento
  if (btnParaPagamento) {
    console.log("Botão para pagamento encontrado"); // Debug
    btnParaPagamento.addEventListener("click", function (e) {
      e.preventDefault();
      console.log("Botão para pagamento clicado"); // Debug

      if (validarFormEntrega()) {
        // Esconde formulários anteriores e mostra formulário de pagamento
        formIdentificacao.style.display = "none";
        formEntrega.style.display = "none";
        formPagamento.style.display = "block";

        // Scroll para o topo do formulário
        window.scrollTo({ top: 0, behavior: "smooth" });
        console.log("Navegando para formulário de pagamento"); // Debug
      }
    });
  } else {
    console.log("Botão para pagamento NÃO encontrado"); // Debug
  }

  // Evento ao clicar no botão de voltar para identificação
  if (btnVoltarParaIdentificacao) {
    btnVoltarParaIdentificacao.addEventListener("click", function (e) {
      e.preventDefault();

      // Mostra formulário de identificação e esconde os outros
      formIdentificacao.style.display = "block";
      formEntrega.style.display = "none";
      formPagamento.style.display = "none";

      // Scroll para o topo do formulário
      window.scrollTo({ top: 0, behavior: "smooth" });
      console.log("Voltando para formulário de identificação"); // Debug
    });
  }

  // Evento ao clicar no botão de voltar para entrega
  if (btnVoltarParaEntrega) {
    btnVoltarParaEntrega.addEventListener("click", function (e) {
      e.preventDefault();

      // Mostra formulário de entrega e esconde os outros
      formIdentificacao.style.display = "none";
      formEntrega.style.display = "block";
      formPagamento.style.display = "none";

      // Scroll para o topo do formulário
      window.scrollTo({ top: 0, behavior: "smooth" });
      console.log("Voltando para formulário de entrega"); // Debug
    });
  }

  // Eventos para alternar entre métodos de pagamento
  if (pixPayment && cardPayment) {
    pixPayment.addEventListener("click", function () {
      pixDataPayment.style.display = "block";
      cardDataPayment.style.display = "none";
      btnFinalizarPix.style.display = "block";
      btnFinalizarCartao.style.display = "none";

      // Adicionar/remover classes ativas
      pixPayment.style.border = "1px solid rgb(46, 133, 236)";
      pixPayment.style.filter =
        "drop-shadow(rgba(46, 133, 236, 0.3) 0px 0px 7px)";
      cardPayment.style.border = "";
      cardPayment.style.filter = "";
    });

    cardPayment.addEventListener("click", function () {
      pixDataPayment.style.display = "none";
      cardDataPayment.style.display = "block";
      btnFinalizarPix.style.display = "none";
      btnFinalizarCartao.style.display = "block";

      // Adicionar/remover classes ativas
      cardPayment.style.border = "1px solid rgb(46, 133, 236)";
      cardPayment.style.filter =
        "drop-shadow(rgba(46, 133, 236, 0.3) 0px 0px 7px)";
      pixPayment.style.border = "";
      pixPayment.style.filter = "";
    });
  }

  // Eventos para os checkboxes "Não tenho e-mail" e "Sem número"
  const noEmailCheckbox = document.getElementById("cld_noEmail");
  const emailInput = document.getElementById("cld_email");

  if (noEmailCheckbox && emailInput) {
    noEmailCheckbox.addEventListener("change", function () {
      if (this.checked) {
        emailInput.disabled = true;
        emailInput.value = "";
        limparErro(emailInput);
      } else {
        emailInput.disabled = false;
      }
    });
  }

  const noNumberCheckbox = document.getElementById("cld_noNumber");
  const numberInput = document.getElementById("cld_number");

  if (noNumberCheckbox && numberInput) {
    noNumberCheckbox.addEventListener("change", function () {
      if (this.checked) {
        numberInput.disabled = true;
        numberInput.value = "";
        limparErro(numberInput);
      } else {
        numberInput.disabled = false;
      }
    });
  }

  // Inicialização: esconder formulários de entrega e pagamento
  if (formEntrega) formEntrega.style.display = "none";
  if (formPagamento) formPagamento.style.display = "none";

  // Inicialmente, mostrar apenas dados do cartão e botão de finalizar compra cartão
  if (pixDataPayment) pixDataPayment.style.display = "none";

  //Como o PIX é a primeira opção de pgto, inicia com o cartão desativado
  if (btnFinalizarCartao) btnFinalizarCartao.style.display = "none";
  //E começa com o PIX ativo
  if (btnFinalizarPix) btnFinalizarPix.style.display = "block";

  // Adicionando listeners para os botões de finalizar compra
  if (btnFinalizarPix) {
    btnFinalizarPix.addEventListener("click", function (e) {
      e.preventDefault();
      // Redireciona para o diretório de PIX no checkout
      // Obter os parâmetros da URL atual
      const currentParams = window.location.search;
      // Redirecionar para /pix mantendo os parâmetros
      window.location.href = "/checkout/pix" + currentParams;
    });
  }

  ///
  ///
  ///
  //VALIDACOES PAGAMENTO
  // Função para validar número de cartão (algoritmo de Luhn)
  function validarNumeroCartao(numero) {
    // Remove espaços e caracteres não numéricos
    numero = numero.replace(/\D/g, "");

    // Verifica o tamanho do número (geralmente entre 13 e 19 dígitos)
    if (numero.length < 13 || numero.length > 19) return false;

    // Implementação do algoritmo de Luhn (Mod 10)
    let soma = 0;
    let dobro = false;

    // Percorre os dígitos da direita para a esquerda
    for (let i = numero.length - 1; i >= 0; i--) {
      let digito = parseInt(numero.charAt(i));

      if (dobro) {
        digito *= 2;
        if (digito > 9) digito -= 9;
      }

      soma += digito;
      dobro = !dobro;
    }

    // O número é válido se a soma for múltiplo de 10
    return soma % 10 === 0;
  }

  // Função para validar data de expiração do cartão
  function validarDataExpiracao(data) {
    if (!data || data.length !== 5) return false;

    const [mes, ano] = data.split("/");
    const mesNum = parseInt(mes);
    const anoNum = parseInt(ano) + 2000; // Assumindo formato MM/YY

    const hoje = new Date();
    const anoAtual = hoje.getFullYear();
    const mesAtual = hoje.getMonth() + 1; // getMonth() retorna 0-11

    // Validação básica do mês
    if (mesNum < 1 || mesNum > 12) return false;

    // Verifica se a data não está no passado
    if (anoNum < anoAtual || (anoNum === anoAtual && mesNum < mesAtual)) {
      return false;
    }

    return true;
  }

  // Função para validar CVV
  function validarCVV(cvv) {
    cvv = cvv.replace(/\D/g, "");
    return cvv.length >= 3 && cvv.length <= 4;
  }

  // Função para validar nome no cartão
  function validarNomeCartao(nome) {
    return nome.trim().length >= 5 && nome.trim().includes(" ");
  }

  // Função principal para validar o formulário de pagamento
  function validarFormPagamento() {
    let valido = true;

    // Verifica qual método de pagamento está ativo
    const pixAtivo =
      document.getElementById("pix_data_payment").style.display !== "none";

    if (pixAtivo) {
      // Para o PIX não há validação adicional
      return true;
    } else {
      // Validar número do cartão
      const numeroCartao = document.getElementById("cld_card_number");
      if (!numeroCartao.value || !validarNumeroCartao(numeroCartao.value)) {
        mostrarErro(numeroCartao, "Cartão inválido. Ex: 1234 5678 9012 3456");
        valido = false;
      } else {
        limparErro(numeroCartao);
      }

      // Validar nome no cartão
      const nomeCartao = document.getElementById("cld_card_name");
      if (!nomeCartao.value || !validarNomeCartao(nomeCartao.value)) {
        mostrarErro(nomeCartao, "Nome inválido. Ex: Pedro Fernandes");
        valido = false;
      } else {
        limparErro(nomeCartao);
      }

      // Validar data de expiração
      const dataExpiracao = document.getElementById("cld_card_expiration_date");
      if (!dataExpiracao.value || !validarDataExpiracao(dataExpiracao.value)) {
        mostrarErro(dataExpiracao, "Vencimento inválido. Ex: 10/25");
        valido = false;
      } else {
        limparErro(dataExpiracao);

        // Preenche os campos ocultos com mês e ano separados
        if (valido) {
          const [mes, ano] = dataExpiracao.value.split("/");
          document.getElementById("cld_card_expiration_date_month").value = mes;
          document.getElementById("cld_card_expiration_date_year").value = ano;
        }
      }

      // Validar CVV
      const cvv = document.getElementById("cld_card_cvv");
      if (!cvv.value || !validarCVV(cvv.value)) {
        mostrarErro(cvv, "Código de segurança inválido. Ex: 123");
        valido = false;
      } else {
        limparErro(cvv);
      }

      // Validar parcelas
      const parcelas = document.getElementById("cld_installments");
      if (!parcelas.value) {
        mostrarErro(parcelas, "Selecione as parcelas");
        valido = false;
      } else {
        limparErro(parcelas);
      }

      return valido;
    }
  }
  //VALIDACOES PAGAMENTO ACIMA
  ///
  ///
  ///
  ///
  // Função para limpar os campos do cartão quando o pagamento for negado
  function limparCamposCartao() {
    // Limpa o número do cartão
    document.getElementById("cld_card_number").value = "";

    // Limpa o nome no cartão
    document.getElementById("cld_card_name").value = "";

    // Limpa a data de validade
    document.getElementById("cld_card_expiration_date").value = "";
    document.getElementById("cld_card_expiration_date_month").value = "";
    document.getElementById("cld_card_expiration_date_year").value = "";

    // Limpa o código de segurança (CVV)
    document.getElementById("cld_card_cvv").value = "";

    // Reseta o select de parcelas para a primeira opção
    const selectParcelas = document.getElementById("cld_installments");
    if (selectParcelas && selectParcelas.options.length > 0) {
      selectParcelas.selectedIndex = 0;
    }

    // Limpa classes de validação que possam ter sido adicionadas
    const camposCartao = [
      "cld_card_number",
      "cld_card_name",
      "cld_card_expiration_date",
      "cld_card_expiration_date_month",
      "cld_card_expiration_date_year",
      "cld_card_cvv",
    ];

    camposCartao.forEach((campo) => {
      const elemento = document.getElementById(campo);
      if (elemento) {
        // Remove classes de validação (se houver)
        elemento.classList.remove("is-valid", "is-invalid");

        // Reseta o estado visual do campo
        elemento.style.borderColor = "";
      }
    });

    // Limpa a seleção da bandeira do cartão
    const brandSelect = document.getElementById("brand-select");
    if (brandSelect) {
      brandSelect.innerHTML = "";
    }
  }
  ///
  ///
  ///
  ///
  // CONTROLE MODAL ABAIXO
  // CONTROLE MODAL ABAIXO
  function openRefusedCardModal() {
    // Seleciona o modal pelo ID
    const modal = document.getElementById("refused-card-modal");
    // Abrir o modal
    modal.style.display = "block";
  }

  const ajaxLoader = document.getElementById("ajax-loader");
  const refusedMessage = document.getElementById("refused-card-message");
  const finalizarButton = document.getElementById(
    "cld_finalize_credit_card_purchase"
  );

  // Função para mostrar o processamento do pagamento
  function mostrarProcessamento() {
    if (ajaxLoader) {
      // Mostra o loader em tela cheia
      ajaxLoader.style.display = "flex";

      // Desativa os botões para evitar cliques duplos
      if (finalizarButton) {
        finalizarButton.disabled = true;
      }
    }
  }

  // Script simplificado para fechar o modal usando ID no botão
  const closeButton = document.getElementById("btn-close-modal");
  const modal = document.getElementById("refused-card-modal");

  // Adiciona evento de clique ao botão
  if (closeButton && modal) {
    closeButton.addEventListener("click", function () {
      // Oculta o modal
      modal.style.display = "none";
      limparCamposCartao();
      refusedMessage.scrollIntoView({
        behavior: "smooth",
        block: "center",
      });
    });
  }

  if (btnFinalizarCartao) {
    btnFinalizarCartao.addEventListener("click", function (e) {
      e.preventDefault();
      console.log("Validando formulário de pagamento...");

      if (validarFormPagamento()) {
        console.log("Formulário de pagamento válido!");
        // Aqui você pode adicionar o código para processar o pagamento
        //alert("Processando pagamento com cartão...");
        //
        //
        // Mostra o loader de processamento
        mostrarProcessamento();

        // Simula o processamento por 5 segundos
        setTimeout(function () {
          // Esconde o loader
          ajaxLoader.style.display = "none";

          // Mostra a mensagem de cartão recusado
          if (refusedMessage) {
            // Remove explicitamente o display:none para garantir que a mensagem apareça
            refusedMessage.style.display = "block";

            // Rola a página até a mensagem de erro
            refusedMessage.scrollIntoView({
              behavior: "smooth",
              block: "center",
            });
            finalizarButton.disabled = false;
            openRefusedCardModal();
          }
        }, 8000);
        //
        //
      } else {
        console.log("Formulário de pagamento inválido!");
      }
    });
  }

  // Adicionar event listeners para os campos de entrada
  document.querySelectorAll("input, select").forEach((campo) => {
    campo.addEventListener("input", function () {
      // Limpa o erro quando o usuário começa a digitar
      limparErro(this);
    });
  });

  // Adicionar o CSS personalizado
  const style = document.createElement("style");
  style.textContent = `
    .cld-invalid-feedback {
      color: #ff0000;
      font-size: 12px;
      margin-top: 5px;
      display: block;
    }
    
    .cld-is-invalid {
      border-color: #ff0000 !important;
    }
    
    input:focus, select:focus {
      outline: none;
    }
  `;
  document.head.appendChild(style);

  // Exibir mensagem quando o script terminar de carregar
  console.log(
    "Script de validação de checkout personalizado carregado com sucesso"
  );
});
//
//
//
//
//
//
//
//
//
// MASCARAS ABAIXO
// MASCARAS ABAIXO
// Função para aplicar máscara a todos os campos
function aplicarMascaras() {
  // Máscara para o campo de telefone
  const telefoneInput = document.getElementById("cld_telephone");
  if (telefoneInput) {
    telefoneInput.addEventListener("input", function (e) {
      let value = this.value.replace(/\D/g, "");

      if (value.length > 0) {
        // Se for celular (11 dígitos)
        if (value.length > 2) {
          value = "(" + value.substring(0, 2) + ") " + value.substring(2);
        }
        if (value.length > 10) {
          // Formato: (99) 99999-9999
          value = value.substring(0, 10) + "-" + value.substring(10);
        }
        if (value.length > 15) {
          value = value.substring(0, 15);
        }
      }

      this.value = value;
    });
  }

  // Máscara para o campo de CPF
  const cpfInput = document.getElementById("cld_document");
  if (cpfInput) {
    cpfInput.addEventListener("input", function (e) {
      let value = this.value.replace(/\D/g, "");

      if (value.length > 0) {
        if (value.length > 3) {
          value = value.substring(0, 3) + "." + value.substring(3);
        }
        if (value.length > 7) {
          value = value.substring(0, 7) + "." + value.substring(7);
        }
        if (value.length > 11) {
          value = value.substring(0, 11) + "-" + value.substring(11);
        }
        if (value.length > 14) {
          value = value.substring(0, 14);
        }
      }

      this.value = value;
    });
  }

  // Máscara para o CEP
  const cepInput = document.getElementById("cld_zip_code");
  if (cepInput) {
    cepInput.addEventListener("input", function (e) {
      let value = this.value.replace(/\D/g, "");

      if (value.length > 5) {
        value = value.substring(0, 5) + "-" + value.substring(5);
      }
      if (value.length > 9) {
        value = value.substring(0, 9);
      }

      this.value = value;
    });
  }

  // Máscara para número do cartão
  const cardNumberInput = document.getElementById("cld_card_number");
  if (cardNumberInput) {
    cardNumberInput.addEventListener("input", function (e) {
      let value = this.value.replace(/\D/g, "");
      let formattedValue = "";

      for (let i = 0; i < value.length; i++) {
        if (i > 0 && i % 4 === 0) {
          formattedValue += " ";
        }
        formattedValue += value[i];
      }

      if (formattedValue.length > 19) {
        formattedValue = formattedValue.substring(0, 19);
      }

      this.value = formattedValue;
    });
  }

  // Máscara para data de expiração do cartão
  const expirationDateInput = document.getElementById(
    "cld_card_expiration_date"
  );
  if (expirationDateInput) {
    expirationDateInput.addEventListener("input", function (e) {
      let value = this.value.replace(/\D/g, "");

      if (value.length > 0) {
        // Limita o mês a valores válidos (01-12)
        if (value.length >= 2) {
          let month = parseInt(value.substring(0, 2));
          if (month > 12) {
            month = 12;
          } else if (month === 0) {
            month = 1;
          }
          value = month.toString().padStart(2, "0") + value.substring(2);
        }

        if (value.length > 2) {
          value = value.substring(0, 2) + "/" + value.substring(2);
        }
        if (value.length > 5) {
          value = value.substring(0, 5);
        }
      }

      this.value = value;

      // Preenche os campos ocultos de mês e ano, se existirem
      if (value.length === 5) {
        const [month, year] = value.split("/");
        const monthInput = document.getElementById(
          "cld_card_expiration_date_month"
        );
        const yearInput = document.getElementById(
          "cld_card_expiration_date_year"
        );

        if (monthInput) monthInput.value = month;
        if (yearInput) yearInput.value = year;
      }
    });
  }

  // Máscara para CVV
  const cvvInput = document.getElementById("cld_card_cvv");
  if (cvvInput) {
    cvvInput.addEventListener("input", function (e) {
      let value = this.value.replace(/\D/g, "");

      if (value.length > 4) {
        value = value.substring(0, 4);
      }

      this.value = value;
    });
  }

  // Máscara para campo de número (endereço)
  const numeroInput = document.getElementById("cld_number");
  if (numeroInput) {
    numeroInput.addEventListener("input", function (e) {
      let value = this.value.replace(/\D/g, "");

      if (value.length > 5) {
        value = value.substring(0, 5);
      }

      this.value = value;
    });
  }
}

// Chame esta função quando o DOM estiver pronto
document.addEventListener("DOMContentLoaded", aplicarMascaras);

//
//
//
//
//
