<?php
/**
 * Classe FacebookConversionAPI
 * 
 * Implementação orientada a objetos para integração com a API de Conversões do Facebook
 * com suporte aprimorado para parâmetros de atribuição
 */
class FacebookConversionAPI {
    private $access_token;
    private $pixel_id;
    private $api_version;
    private $debug_mode;
    private $test_event_code;
    
    /**
     * Construtor
     * 
     * @param string $access_token Token de acesso da API
     * @param string $pixel_id ID do Pixel do Facebook
     * @param string $api_version Versão da API (opcional)
     * @param bool $debug_mode Modo de depuração (opcional)
     * @param string $test_event_code Código de evento de teste (opcional)
     */
    public function __construct($access_token, $pixel_id, $api_version = 'v18.0', $debug_mode = false, $test_event_code = null) {
        $this->access_token = $access_token;
        $this->pixel_id = $pixel_id;
        $this->api_version = $api_version;
        $this->debug_mode = $debug_mode;
        $this->test_event_code = $test_event_code;
    }
    
    /**
     * Envia um evento de visualização de página
     * 
     * @param array $page_data Dados da página
     * @param array $user_data Dados do usuário
     * @param array $custom_data Dados personalizados adicionais
     * @param string $event_id ID do evento personalizado (opcional)
     * @return array Resposta da API
     */
    public function sendPageViewEvent($page_data, $user_data, $custom_data = [], $event_id = null) {
        // Se não for fornecido um event_id, gera um usando a URL da página ou session_id
        if ($event_id === null) {
            $page_url = $page_data['page_url'] ?? ($_SERVER['REQUEST_URI'] ?? '');
            $unique_id = md5($page_url . (session_id() ?: uniqid()));
            $event_id = FacebookAttributionHelper::generateEventId('PageView', $unique_id);
        }
        
        return $this->sendEvent('PageView', $page_data, $user_data, $custom_data, $event_id);
    }

    /**
     * Envia um evento de compra para a API de Conversões
     * 
     * @param array $purchase_data Dados da compra
     * @param array $user_data Dados do usuário
     * @param array $custom_data Dados personalizados adicionais
     * @param string $event_id ID do evento personalizado (opcional)
     * @return array Resposta da API
     */
    public function sendPurchaseEvent($purchase_data, $user_data, $custom_data = [], $event_id = null) {
        // Se não for fornecido um event_id, gera um usando o order_id
        if ($event_id === null && isset($purchase_data['order_id'])) {
            $event_id = FacebookAttributionHelper::generateEventId('Purchase', $purchase_data['order_id']);
        }
        
        return $this->sendEvent('Purchase', $purchase_data, $user_data, $custom_data, $event_id);
    }
    
    /**
     * Envia um evento de adição ao carrinho
     * 
     * @param array $cart_data Dados do carrinho
     * @param array $user_data Dados do usuário
     * @param array $custom_data Dados personalizados adicionais
     * @param string $event_id ID do evento personalizado (opcional)
     * @return array Resposta da API
     */
    public function sendAddToCartEvent($cart_data, $user_data, $custom_data = [], $event_id = null) {
        // Se não for fornecido um event_id, gera um usando o cart_id ou session_id
        if ($event_id === null) {
            $unique_id = $cart_data['cart_id'] ?? (session_id() ?: uniqid());
            $event_id = FacebookAttributionHelper::generateEventId('AddToCart', $unique_id);
        }
        
        return $this->sendEvent('AddToCart', $cart_data, $user_data, $custom_data, $event_id);
    }
    
    /**
     * Envia um evento de início de checkout
     * 
     * @param array $checkout_data Dados do checkout
     * @param array $user_data Dados do usuário
     * @param array $custom_data Dados personalizados adicionais
     * @param string $event_id ID do evento personalizado (opcional)
     * @return array Resposta da API
     */
    public function sendInitiateCheckoutEvent($checkout_data, $user_data, $custom_data = [], $event_id = null) {
        // Se não for fornecido um event_id, gera um usando session_id
        if ($event_id === null) {
            $unique_id = $checkout_data['checkout_id'] ?? (session_id() ?: uniqid());
            $event_id = FacebookAttributionHelper::generateEventId('InitiateCheckout', $unique_id);
        }
        
        return $this->sendEvent('InitiateCheckout', $checkout_data, $user_data, $custom_data, $event_id);
    }
    
    /**
     * Método genérico para enviar qualquer tipo de evento
     * 
     * @param string $event_name Nome do evento
     * @param array $event_data Dados principais do evento
     * @param array $user_data Dados do usuário
     * @param array $custom_data Dados personalizados adicionais
     * @param string $event_id ID do evento personalizado (opcional)
     * @return array Resposta da API
     */
    public function sendEvent($event_name, $event_data, $user_data, $custom_data = [], $event_id = null) {
        // Endpoint da API de Conversões
        $url = "https://graph.facebook.com/{$this->api_version}/{$this->pixel_id}/events";
        
        // Se não for fornecido um event_id, gera um padrão
        if ($event_id === null) {
            $event_id = strtolower($event_name) . '_' . uniqid();
        }
        
        // Captura automaticamente os parâmetros de atribuição se não fornecidos
        if (!isset($user_data['fbc'])) {
            $user_data['fbc'] = FacebookAttributionHelper::captureFbclid();
        }
        
        if (!isset($user_data['fbp'])) {
            $user_data['fbp'] = FacebookAttributionHelper::getFbp();
        }
        
        // Prepara o hash dos dados do usuário
        $hashed_user_data = $this->hashUserData($user_data);
        
        // Prepara os dados do evento
        $event_time = time();
        
        // Prepara os dados customizados com base no tipo de evento
        $prepared_custom_data = $this->prepareCustomData($event_name, $event_data, $custom_data);
        
        // Prepara o payload com os dados do evento
        $data = [
            'data' => [
                [
                    'event_name' => $event_name,
                    'event_time' => $event_time,
                    'event_id' => $event_id,
                    'event_source_url' => $event_data['source_url'] ?? $_SERVER['HTTP_REFERER'] ?? '',
                    'action_source' => 'website',
                    'user_data' => $hashed_user_data,
                    'custom_data' => $prepared_custom_data,
                ]
            ],
            'access_token' => $this->access_token,
        ];
        
        // Se o modo de depuração estiver ativado e tiver um código de evento de teste, inclui-o
        if ($this->debug_mode && $this->test_event_code) {
            $data['test_event_code'] = $this->test_event_code;
        }
        
        // Configura a requisição cURL
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
        
        // Executa a requisição
        $response = curl_exec($ch);
        $error = curl_error($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        // Registra o evento para depuração se necessário
        if ($this->debug_mode) {
            $this->logEvent($event_name, $data, $response, $http_code);
        }

        //independente do debug mode salva log
        $this->logEvent($event_name, $data, $response, $http_code);
        
        // Trata a resposta
        if ($error) {
            return [
                'success' => false,
                'message' => "Erro cURL: $error",
                'http_code' => $http_code
            ];
        }
        
        $response_data = json_decode($response, true);
        return [
            'success' => isset($response_data['events_received']) && $response_data['events_received'] > 0,
            'data' => $response_data,
            'http_code' => $http_code,
            'event_id' => $event_id // Retorna o event_id para referência
        ];
    }
    
    /**
     * Prepara os dados customizados com base no tipo de evento
     * 
     * @param string $event_name Nome do evento
     * @param array $event_data Dados do evento
     * @param array $custom_data Dados personalizados adicionais
     * @return array Dados customizados formatados
     */
    private function prepareCustomData($event_name, $event_data, $custom_data = []) {
        $prepared_data = [];
        
        switch ($event_name) {
            case 'Purchase':
                $prepared_data = [
                    'currency' => $event_data['currency'] ?? 'BRL',
                    'value' => $event_data['value'] ?? 0,
                    'content_ids' => $event_data['product_ids'] ?? [],
                    'content_type' => 'product',
                    'order_id' => $event_data['order_id'] ?? '',
                ];
                break;
                
            case 'AddToCart':
                $prepared_data = [
                    'currency' => $event_data['currency'] ?? 'BRL',
                    'value' => $event_data['value'] ?? 0,
                    'content_ids' => $event_data['product_ids'] ?? [],
                    'content_type' => 'product',
                    'contents' => $event_data['contents'] ?? [],
                ];
                break;
                
            case 'InitiateCheckout':
                $prepared_data = [
                    'currency' => $event_data['currency'] ?? 'BRL',
                    'value' => $event_data['value'] ?? 0,
                    'content_ids' => $event_data['product_ids'] ?? [],
                    'content_type' => 'product',
                    'num_items' => $event_data['num_items'] ?? 0,
                ];
                break;
                
            case 'ViewContent':
                $prepared_data = [
                    'currency' => $event_data['currency'] ?? 'BRL',
                    'value' => $event_data['value'] ?? 0,
                    'content_ids' => $event_data['product_ids'] ?? [],
                    'content_type' => 'product',
                    'content_name' => $event_data['content_name'] ?? '',
                    'content_category' => $event_data['content_category'] ?? '',
                ];
                break;
            
            case 'PageView':
                $prepared_data = [
                    'page_url' => $event_data['page_url'] ?? ($_SERVER['REQUEST_URI'] ?? ''),
                    'page_title' => $event_data['page_title'] ?? '',
                    'page_referrer' => $event_data['page_referrer'] ?? ($_SERVER['HTTP_REFERER'] ?? ''),
                ];
                break;
                
            // Adicione outros casos conforme necessário
            
            default:
                // Para eventos genéricos, apenas passamos os dados como estão
                $prepared_data = [
                    'currency' => $event_data['currency'] ?? 'BRL',
                    'value' => $event_data['value'] ?? 0,
                ];
                break;
        }
        
        // Mescla com os dados personalizados adicionais
        return array_merge($prepared_data, $custom_data);
    }
    
    /**
     * Cria hash dos dados do usuário conforme requerido pelo Facebook
     * 
     * @param array $user_data Dados do usuário
     * @return array Dados do usuário com hash
     */
    private function hashUserData($user_data) {
        $hashed_data = [];
        
        // Normaliza e faz hash do e-mail
        if (isset($user_data['email'])) {
            $email = strtolower(trim($user_data['email']));
            $hashed_data['em'] = hash('sha256', $email);
        }
        
        // Normaliza e faz hash do telefone (remove caracteres não numéricos)
        if (isset($user_data['phone'])) {
            $phone = preg_replace('/[^0-9]/', '', $user_data['phone']);
            $hashed_data['ph'] = hash('sha256', $phone);
        }
        
        // Normaliza e faz hash do nome
        if (isset($user_data['first_name'])) {
            $fn = strtolower(trim($user_data['first_name']));
            $hashed_data['fn'] = hash('sha256', $fn);
        }
        
        if (isset($user_data['last_name'])) {
            $ln = strtolower(trim($user_data['last_name']));
            $hashed_data['ln'] = hash('sha256', $ln);
        }
        
        // Cidade, estado, CEP
        if (isset($user_data['city'])) {
            $hashed_data['ct'] = hash('sha256', strtolower(trim($user_data['city'])));
        }
        
        if (isset($user_data['state'])) {
            $hashed_data['st'] = hash('sha256', strtolower(trim($user_data['state'])));
        }
        
        if (isset($user_data['zip'])) {
            $hashed_data['zp'] = hash('sha256', $user_data['zip']);
        }
        
        // Endereço IP e User Agent (opcional, mas recomendado para melhor matching)
        if (isset($user_data['client_ip_address'])) {
            $hashed_data['client_ip_address'] = $user_data['client_ip_address'];
        } else {
            $hashed_data['client_ip_address'] = $_SERVER['REMOTE_ADDR'] ?? null;
        }
        
        if (isset($user_data['client_user_agent'])) {
            $hashed_data['client_user_agent'] = $user_data['client_user_agent'];
        } else {
            $hashed_data['client_user_agent'] = $_SERVER['HTTP_USER_AGENT'] ?? null;
        }
        
        // Facebook Click ID para melhor atribuição
        if (isset($user_data['fbc']) && $user_data['fbc']) {
            $hashed_data['fbc'] = $user_data['fbc'];
        }
        
        if (isset($user_data['fbp']) && $user_data['fbp']) {
            $hashed_data['fbp'] = $user_data['fbp'];
        }
        
        // Identificador externo (opcional)
        if (isset($user_data['external_id'])) {
            $hashed_data['external_id'] = hash('sha256', $user_data['external_id']);
        }
        
        return $hashed_data;
    }
    
    /**
     * Registra eventos para depuração
     * 
     * @param string $event_name Nome do evento
     * @param array $request_data Dados da requisição
     * @param string $response_data Dados da resposta
     * @param int $http_code Código HTTP da resposta
     */
    private function logEvent($event_name, $request_data, $response_data, $http_code) {
        $log_dir = __DIR__ . '/logs';
        
        // Cria o diretório de logs se não existir
        if (!file_exists($log_dir)) {
            mkdir($log_dir, 0755, true);
        }
        
        $log_file = $log_dir . '/fb_events_' . date('Y-m-d') . '.log';
        
        // Sanitiza os dados de requisição removendo o token de acesso
        $sanitized_request = $request_data;
        $sanitized_request['access_token'] = 'REDACTED';
        
        $log_data = [
            'timestamp' => date('Y-m-d H:i:s'),
            'event_name' => $event_name,
            'request' => $sanitized_request,
            'response' => json_decode($response_data, true),
            'http_code' => $http_code
        ];
        
        file_put_contents(
            $log_file, 
            json_encode($log_data, JSON_PRETTY_PRINT) . "\n\n", 
            FILE_APPEND
        );
    }
}