// checkout.js - Script otimizado para checkout PIX com correções de carrinho

// Variáveis globais
let qrcodeData;
let idTransacaoData;
let customerData;
let carrinho;
let dadosForamEnviados = false;
let userNome,
  userEmail,
  userCpf,
  userTelefone,
  userUtmSource,
  userUtmCampaign,
  userUtmMedium,
  userUtmContent,
  userUtmTerm,
  userSrc;

// Inicializar quando o DOM estiver carregado
document.addEventListener("DOMContentLoaded", function () {
  console.log("DOM carregado, inicializando...");
  recuperarDadosUsuario();
  inicializarCarrinho();
  inicializarTimer();

  // Iniciar processamento do PIX
  if (carrinho && carrinho.total) {
    var valorCarrinho = Number((carrinho.total * 100).toFixed(0));
    console.log("Valor do carrinho em CENTAVOS: " + valorCarrinho);
    atualizarValorNoSpan(valorCarrinho);

    var dataToQRcode = {
      valor: valorCarrinho,
      nome: userNome,
      email: userEmail,
      doc_tipo: "cpf",
      doc_num: userCpf,
    };

    gerarQRCode(dataToQRcode);
  } else {
    console.error("Carrinho não inicializado corretamente");
    mostrarNotificacao("Erro ao carregar dados do carrinho. Tente novamente.");
  }

  // Configurar listener para o botão de cópia
  document
    .getElementById("copy_digitable_line")
    .addEventListener("click", copiarCodigo);
});

/**
 * Sanitiza input para evitar XSS
 * @param {string} input - Texto para sanitizar
 * @returns {string} - Texto sanitizado
 */
function sanitizeInput(input) {
  if (!input) return "";

  // Criar elemento temporário para sanitização
  const temp = document.createElement("div");
  temp.textContent = input;
  return temp.innerHTML;
}

/**
 * Recupera dados do usuário do localStorage de forma segura
 */
function recuperarDadosUsuario() {
  console.log("Iniciando recuperação de dados do usuário...");

  // Recuperar e sanitizar dados
  userNome = sanitizeInput(localStorage.getItem("nome")) || "Nome na Fila";
  userEmail =
    sanitizeInput(localStorage.getItem("email")) ||
    "email_alternativo@example.com";
  userCpf = sanitizeInput(localStorage.getItem("cpf")) || "00938585231";
  userTelefone =
    sanitizeInput(localStorage.getItem("telefone")) || "21999994033";
  userUtmSource = sanitizeInput(localStorage.getItem("utmSource")) || "";
  userUtmCampaign = sanitizeInput(localStorage.getItem("utmCampaign")) || "";
  userUtmMedium = sanitizeInput(localStorage.getItem("utmMedium")) || "";
  userUtmContent = sanitizeInput(localStorage.getItem("utmContent")) || "";
  userUtmTerm = sanitizeInput(localStorage.getItem("utmTerm")) || "";
  userSrc = sanitizeInput(localStorage.getItem("src")) || "";

  console.log("Usuário recuperado:", userNome, userEmail);

  // Criar objeto de dados do cliente para uso posterior
  customerData = {
    name: userNome,
    email: userEmail,
    phone: userTelefone,
    document: userCpf,
    country: "BR",
    trackingParameters: {
      utm_source: userUtmSource,
      utm_campaign: userUtmCampaign,
      utm_medium: userUtmMedium,
      utm_content: userUtmContent,
      utm_term: userUtmTerm,
      src: userSrc,
    },
  };
}

/**
 * Obtém e inicializa os dados do carrinho
 */
function inicializarCarrinho() {
  console.log("Inicializando carrinho...");

  try {
    const carrinhoSalvo = localStorage.getItem("wepinkCarrinho");
    if (carrinhoSalvo) {
      carrinho = JSON.parse(carrinhoSalvo);
    } else {
      // Carrinho padrão se não existir
      carrinho = {
        itens: [],
        total: 0,
        desconto: 0,
      };
    }

    // Garantir que o carrinho tenha uma estrutura válida
    if (!carrinho.itens) carrinho.itens = [];
    if (isNaN(carrinho.total)) carrinho.total = 0;
    if (isNaN(carrinho.desconto)) carrinho.desconto = 0;

    console.log("Carrinho inicializado:", carrinho);

    // Adicionar o carrinho aos dados do cliente
    if (customerData) {
      customerData.cart = carrinho;
    }
  } catch (error) {
    console.error("Erro ao inicializar carrinho:", error);
    carrinho = {
      itens: [],
      total: 0,
      desconto: 0,
    };
  }
}

function logError(tipoErro, mensagemErro) {
  const data = new Date();
  const timestamp = data.toLocaleString("pt-BR", { timeZone: "America/Sao_Paulo" }); // Timezone de São Paulo
  const erro = {
    tipo: tipoErro,
    mensagem: mensagemErro,
    dataHora: timestamp,
  };

  fetch("/conv/ErrorHandler.php", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify(erro),
  })
  .then((response) => {
    if (response.ok) {
      console.log("[Log] - Erro registrado com sucesso.");
    } else {
      console.error("[Log] - Falha ao registrar erro.");
    }
  })
  .catch((error) => {
    console.error("[Log] - Erro ao comunicar com ErrorHandler:", error);
  });
}

/**
 * Inicializa o timer de contagem regressiva
 */
function inicializarTimer() {
  let timeInSeconds = 600; // 10 minutos
  const timerElement = document.getElementById("timer");
  let timerInterval;

  function updateTimer() {
    const minutes = Math.floor(timeInSeconds / 60);
    const seconds = timeInSeconds % 60;

    const formattedMinutes = String(minutes).padStart(2, "0");
    const formattedSeconds = String(seconds).padStart(2, "0");

    timerElement.textContent = `${formattedMinutes}:${formattedSeconds}`;

    timeInSeconds--;

    if (timeInSeconds < 0) {
      clearInterval(timerInterval);
      timerElement.textContent = "Expirado";
      timerElement.classList.add("text-red-700");
      mostrarNotificacao(
        "O tempo para pagamento expirou. Atualize a página para gerar um novo código."
      );

      // Rastrear expiração do PIX
      fbq("trackCustom", "PIXExpired", {
        transaction_id: idTransacaoData,
      });
    }
  }

  // Atualizar inicialmente
  updateTimer();

  // Configurar intervalo
  timerInterval = setInterval(updateTimer, 1000);
}

/**
 * Gera o código PIX
 * @param {Object} dataToQRcode - Dados para gerar o PIX
 */
function gerarQRCode(dataToQRcode) {
  console.log("Gerando código PIX...");

  fbq("trackCustom", "PIXGenerating");

  fetch("ativopay/qrcode.php", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify(dataToQRcode),
  })
    .then((response) => {
      if (!response.ok) {
        throw new Error(`Erro ao gerar QR Code: ${response.statusText}`);
      }
      return response.json();
    })
    .then((response) => {
      console.log("QR Code gerado com sucesso");
      console.log(response);
      
      console.log("Before sendOrderDataWaitingPayment()");
      sendOrderDataWaitingPayment();
      console.log("After sendOrderDataWaitingPayment()");

      qrcodeData = response.qrcode;
      idTransacaoData = response.id;

      // Atualizar elementos da UI
      document.getElementById("key_pix_digitable_line1").value = qrcodeData;
      document.getElementById("key_pix_digitable_line2").value = qrcodeData;

      // Habilitar botão de cópia
      const copyButton = document.getElementById("copy_digitable_line");
      copyButton.classList.remove("bg-[#d3d3d3]");
      copyButton.classList.add("bg-[#1FBB70]");
      copyButton.disabled = false;

      // Rastrear evento de PIX gerado
      fbq("trackCustom", "PIXGenerated", {
        transaction_id: idTransacaoData,
        value: carrinho.total,
      });

      // Iniciar verificação de status
      const dataToStatus = {
        id_transacao: idTransacaoData,
      };

      verificarStatusPagamento(dataToStatus);
    })
    .catch((error) => {
      console.error("Erro ao gerar código PIX:", error);
      mostrarNotificacao(
        "Erro ao gerar o código PIX. Tente novamente atualizando a página."
      );

      // Rastrear erro
      fbq("trackCustom", "PIXError", {
        error_type: "fetch_error",
        error_message: error.message,
      });
    });
}
/**
 * Verifica o status do pagamento PIX
 * @param {Object} dataToStatus - Dados para verificação
 */
 //NOVO 12/09/2025 - 01:48
function verificarStatusPagamento(dataToStatus) {
  let intervaloInicial = 2000; // 2 segundos
  let intervaloMaximo = 10000; // 10 segundos
  let intervaloAtual = intervaloInicial;
  let verificacaoAtiva = true; // Flag para controlar o polling

  function verificarStatus() {
    // Se já foi processado, parar todas as verificações
    if (dadosForamEnviados || !verificacaoAtiva) {
      console.log("Verificação interrompida - pagamento já processado");
      return;
    }

    fetch("ativopay/status.php", {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
      },
      body: JSON.stringify(dataToStatus),
    })
      .then((response) => {
        if (!response.ok) {
          return response.json().then((errorData) => {
            const errorCode = response.status;
            const errorMessage = errorData?.message || response.statusText;

            console.error(`Erro ${errorCode}: ${errorMessage}`);
            logError("Erro ao verificar status", `Código: ${errorCode}, Mensagem: ${errorMessage}`);

            throw new Error(`Código: ${errorCode}, Mensagem: ${errorMessage}`);
          });
        }
        return response.json();
      })
      .then((response) => {
        console.log("response eh", response);
        var statusPagamento = response?.data?.status;
        console.log("statusPagamento eh: ", statusPagamento);

        if (statusPagamento === "PAID") {
          console.log("Pagamento aprovado!");

          // PARAR IMEDIATAMENTE todas as verificações futuras
          verificacaoAtiva = false;

          // Verificar E marcar atomicamente
          if (!dadosForamEnviados) {
            dadosForamEnviados = true; // Marcar IMEDIATAMENTE
            
            console.log("Processando pagamento - timestamp:", new Date().toISOString());

            // Executar as operações
            try {
              realizarPurchase();
              console.log("realizarPurchase executado");
            } catch (error) {
              console.error("Erro ao realizar compra:", error);
              logError("Erro ao realizar compra [realizarPurchase]", error.message);
            }

            // Delay de 100ms antes do sendOrderData
            setTimeout(() => {
              try {
                sendOrderData();
              } catch (error) {
                console.error("Erro ao enviar dados do pedido:", error);
                logError("Metodo (verificarStatusPagamento) - /checkout/pix/js/checkout.js] - Erro ao enviar dados do pedido [sendOrderData]", error.message);
              }
            }, 100);

            // Redirecionar
            setTimeout(() => {
              window.location.href = "/frete";
            }, 1500);

          } else {
                console.log("Pagamento já foi processado anteriormente");
                console.log("Redirecionando usuário para área de frete...");
                setTimeout(() => {
                    window.location.href = "/frete";
                }, 1000); // Delay menor já que não precisa processar nada
          }

        } else {
          console.log("Aguardando pagamento...");
          // Só continuar se ainda estiver ativo
          if (verificacaoAtiva && !dadosForamEnviados) {
            intervaloAtual = Math.min(intervaloAtual * 1.5, intervaloMaximo);
            setTimeout(verificarStatus, intervaloAtual);
          }
        }
      })
      .catch((error) => {
        const errorMessage = `Erro ao verificar status: ${error.message}`;
        console.error(errorMessage);
        logError("Erro ao verificar status", errorMessage);

        // Só continuar se ainda estiver ativo
        if (verificacaoAtiva && !dadosForamEnviados) {
          intervaloAtual = Math.min(intervaloAtual * 2, intervaloMaximo);
          setTimeout(verificarStatus, intervaloAtual);
        }
      });
  }

  // Iniciar verificação
  verificarStatus();
}

function realizarPurchase() {
  // Obter dados do carrinho
  let carrinho = localStorage.getItem("wepinkCarrinho");
  if (carrinho) {
    carrinho = JSON.parse(carrinho);
  } else {
    // Carrinho padrão se não existir
    carrinho = {
      itens: [],
      total: 0,
      desconto: 0,
    };
  }

  // Se o carrinho estiver vazio, apenas registrar no console e prosseguir
  if (!carrinho.itens || carrinho.itens.length === 0) {
    console.error("Carrinho vazio");
    logError("Carrinho vazio", "Carrinho vazio na função realizarPurchase.");
    return; // Sai da função sem executar o restante
  }

  // Obter dados do usuário do localStorage
  const nome = localStorage.getItem("nome") || "";
  const email = localStorage.getItem("email") || "";
  const telefone = localStorage.getItem("telefone") || "";

  // Adicionar o prefixo 55 ao telefone se não existir
  const phoneFormatado = telefone.startsWith("55") ? telefone : `55${telefone}`;

  // Criar a estrutura do JSON a ser enviado
  const dadosEnvio = {
    ...carrinho, // Adiciona todos os campos do carrinho (itens, total, desconto)
    user: {
      email: email,
      phone: phoneFormatado,
      full_name: nome,
    },
    current_url: window.location.href, // Adiciona a URL atual
  };

  console.log("[Purchase] - Dados que serão enviados:", dadosEnvio);

  // Fazer a requisição para a API
  fetch("/conv/Integration.php?action=purchase", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify(dadosEnvio),
  })
    .then((response) => {
      if (!response.ok) {
        throw new Error(`[Purchase] - Erro na requisição: ${response.status}`);
      }
      return response.json();
    })
    .then((data) => {
      console.log("[Purchase] - Resposta da API:", data);
      // Aqui você pode processar a resposta conforme necessário
      if (data.status === "success") {
        console.log("[Purchase] - Enviado com sucesso!");
        // Redirecionar ou atualizar a interface se necessário
      } else {
        console.error("[Purchase] - Erro ao enviar:", data.message);
        logError("Erro ao enviar", data.message);
      }
    })
    .catch((error) => {
      console.error("[Purchase] - Erro ao enviar dados para API:", error);
      logError("Erro ao enviar dados para API", error.message);
    });
}

/**
 * Envia os dados do pedido para o servidor
 */
function sendOrderData() {
  console.log("Enviando dados do pedido...");
  
  fetch("Service/Payment/MyPixProcessor.php?status=paid", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify(customerData),
  })
    .then((response) => {
      if (!response.ok) {
        throw new Error(`Erro ao enviar dados do pedido: ${response.statusText}`);
      }
      console.log("passei aqui xD");
      return response.text(); // ou .json() se o backend retornar JSON
    })
    .then((data) => {
      console.log("Dados do pedido enviados com sucesso:", data);
    })
    .catch((error) => {
      console.error("Erro ao enviar dados do pedido:", error);
      // Chama a função de log para registrar o erro no servidor
      logError("Erro ao enviar dados do pedido", error.message);
    });
}

function sendOrderDataWaitingPayment() {
  console.log("Enviando dados do pedido waiting payment...");
  fetch("Service/Payment/MyPixProcessor.php?status=waiting_payment", {
    method: "POST",
    headers: {
      "Content-Type": "application/json",
    },
    body: JSON.stringify(customerData),
  })
    .then((response) => {
      if (!response.ok) {
        throw new Error(
          `Erro ao enviar dados do pedido: ${response.statusText}`
        );
      }
      console.log("passei aqui xD");
      return response.text(); // ou .json() se o backend retornar JSON
    })
    .then((data) => {
      console.log("Dados do pedido enviados com sucesso:", data);
    })
    .catch((error) => {
      console.error("Erro ao enviar dados do pedido:", error);
    });
}


/**
 * Copia o código PIX para o clipboard
 */
function copiarCodigo() {
  if (!qrcodeData) {
    console.error("Código PIX não disponível");
    return;
  }

  // Tentar usar o clipboard moderno
  if (navigator.clipboard) {
    navigator.clipboard
      .writeText(qrcodeData)
      .then(() => {
        mostrarNotificacao(
          "Código PIX copiado! Use-o no seu banco para pagar."
        );

        // Rastrear evento de cópia
        fbq("trackCustom", "PIXCodeCopied", {
          transaction_id: idTransacaoData,
        });
      })
      .catch((err) => {
        console.error("Erro ao copiar código:", err);
        usarFallbackCopy();
      });
  } else {
    // Fallback para navegadores mais antigos
    usarFallbackCopy();
  }

  // Método de cópia alternativo
  function usarFallbackCopy() {
    const input = document.createElement("textarea");
    input.value = qrcodeData;
    document.body.appendChild(input);
    input.select();

    try {
      document.execCommand("copy");
      mostrarNotificacao("Código PIX copiado! Use-o no seu banco para pagar.");

      // Rastrear evento de cópia
      fbq("trackCustom", "PIXCodeCopied", {
        transaction_id: idTransacaoData,
      });
    } catch (err) {
      console.error("Erro ao copiar código (fallback):", err);
      mostrarNotificacao(
        "Não foi possível copiar automaticamente. Selecione e copie o código manualmente."
      );
    }

    document.body.removeChild(input);
  }
}

/**
 * Exibe uma notificação na tela
 * @param {string} mensagem - Mensagem a ser exibida
 * @param {number} duracao - Duração em ms (padrão: 3000)
 */
function mostrarNotificacao(mensagem, duracao = 3000) {
  const notificacao = document.getElementById("notification");

  if (!notificacao) {
    // Criar elemento de notificação se não existir
    const novaNotificacao = document.createElement("div");
    novaNotificacao.id = "notification";
    novaNotificacao.className = "notification";
    document.body.appendChild(novaNotificacao);

    setTimeout(() => {
      mostrarNotificacao(mensagem, duracao);
    }, 100);
    return;
  }

  // Limpar timeout anterior
  if (notificacao.timeoutId) {
    clearTimeout(notificacao.timeoutId);
  }

  // Configurar e exibir notificação
  notificacao.textContent = mensagem;
  notificacao.style.display = "block";

  // Ocultar após a duração
  notificacao.timeoutId = setTimeout(() => {
    notificacao.style.display = "none";
  }, duracao);
}

/**
 * Atualiza o valor formatado no elemento da UI
 * @param {number} valorInteiro - Valor em centavos
 */
function atualizarValorNoSpan(valorInteiro) {
  // Dividir por 100 para converter de centavos para reais
  const valorEmReais = valorInteiro / 100;

  // Formatar com Intl.NumberFormat
  const valorFormatado = new Intl.NumberFormat("pt-BR", {
    style: "currency",
    currency: "BRL",
  }).format(valorEmReais);

  // Atualizar elemento
  document.getElementById("valor-total-pagamento").textContent = valorFormatado;
}

// Proteção contra inspeção do browser
/*
(function () {
  // Desabilitar clique do botão direito
  document.addEventListener("contextmenu", (e) => {
    e.preventDefault();
    return false;
  });

  // Desabilitar atalhos de desenvolvedor
  document.addEventListener("keydown", (e) => {
    const disabledKeys = ["c", "C", "x", "J", "u", "I"];
    if ((e.ctrlKey && disabledKeys.includes(e.key)) || e.key === "F12") {
      e.preventDefault();
      return false;
    }
  });
})();
*/
