// Função para atualizar o select de parcelas com base no valor total do carrinho
function atualizarOpcoesParcelas() {
  console.log("executing atualizarOpcoesParcelas()");
  // Obtém o valor total do carrinho
  const carrinhoSalvo = localStorage.getItem("wepinkCarrinho");
  let valorTotal = 0;

  if (carrinhoSalvo) {
    const carrinho = JSON.parse(carrinhoSalvo);
    valorTotal = carrinho.total;
  }

  // Valor mínimo para garantir que sempre haja um valor a ser exibido
  if (valorTotal <= 0) valorTotal = 0.0; // Valor padrão visto no código original

  // Referência ao elemento select
  const selectParcelas = document.getElementById("cld_installments");
  if (!selectParcelas) return;

  // Limpa as opções existentes
  selectParcelas.innerHTML = "";

  // Taxa de juros para parcelamento (3.99% ao mês conforme indicado no footer) - DESATIVADO!
  const taxaMensal = 0.0;

  // Cria as novas opções de 1x a 12x
  for (let i = 1; i <= 12; i++) {
    const option = document.createElement("option");
    option.value = i;

    // Se for à vista, não aplica juros
    if (i === 1) {
      option.textContent = `1 x R$ ${formatarMoeda(valorTotal)}`;
      option.selected = true; // Seleciona por padrão
    } else {
      // Cálculo do valor da parcela com juros
      const valorParcela = calcularValorParcela(valorTotal, i, taxaMensal);
      option.textContent = `${i} x R$ ${formatarMoeda(valorParcela)}`;
    }

    // Adiciona a opção ao select
    selectParcelas.appendChild(option);
  }
}

// Calcula o valor da parcela com juros
function calcularValorParcela(valorTotal, numParcelas, taxaMensal) {
  if (taxaMensal === 0 || numParcelas === 1) {
    // Se não há juros ou é pagamento à vista, apenas divide o valor total
    return valorTotal / numParcelas;
  } else {
    // Caso contrário, usa a fórmula de juros compostos
    return (
      (valorTotal * (taxaMensal * Math.pow(1 + taxaMensal, numParcelas))) /
      (Math.pow(1 + taxaMensal, numParcelas) - 1)
    );
  }
}

// Formata valor para exibição em moeda brasileira
function formatarMoeda(valor) {
  return valor.toLocaleString("pt-BR", {
    minimumFractionDigits: 2,
    maximumFractionDigits: 2,
  });
}

// Observa mudanças no carrinho e atualiza parcelas
function observarCarrinho() {
  // Salva a função original
  const funcaoRenderizarOriginal = window.renderizarCarrinho;

  if (typeof funcaoRenderizarOriginal === "function") {
    // Substitui a função por uma versão que também atualiza as parcelas
    window.renderizarCarrinho = function () {
      // Chama a função original primeiro
      funcaoRenderizarOriginal.apply(this, arguments);
      // Então atualiza as parcelas
      atualizarOpcoesParcelas();
    };
  }

  // Também atualiza quando a página é carregada
  atualizarOpcoesParcelas();
}

// Executa quando o DOM estiver pronto
document.addEventListener("DOMContentLoaded", function () {
  // Aguarda um momento para garantir que todas as outras funções da página já foram carregadas
  setTimeout(function () {
    observarCarrinho();
    // Executa imediatamente na primeira carga
    atualizarOpcoesParcelas();
  }, 500);
});
