// Estrutura para armazenar os produtos
let carrinho = {
  itens: [],
  total: 0,
  desconto: 0,
};

// Script para integração de OrderBump com Carrinho

document.addEventListener("DOMContentLoaded", function () {
  // Configuração inicial do OrderBump
  configurarOrderBump();

  // Inicializa o carrinho
  inicializarCarrinho();
});

// Extrai o valor numérico de uma string de preço (ex: "R$29,90" -> 29.90)
function extrairValorNumerico(valorString) {
  // Remove "R$" e substitui vírgula por ponto
  return parseFloat(valorString.replace("R$", "").replace(",", ".").trim());
}

// Função para exibir a mensagem de limite excedido
function exibirMensagemLimite() {
  // Cria o elemento de mensagem se não existir
  let mensagemElement = document.getElementById("mensagem-limite");

  if (!mensagemElement) {
    mensagemElement = document.createElement("div");
    mensagemElement.id = "mensagem-limite";
    mensagemElement.className =
      "fixed top-4 right-4 bg-red-500 text-white p-4 rounded-md shadow-lg z-50";
    mensagemElement.style.maxWidth = "300px";
    mensagemElement.style.opacity = "0";
    mensagemElement.style.transition = "opacity 0.3s ease";
    document.body.appendChild(mensagemElement);
  }

  // Define o conteúdo e exibe a mensagem
  mensagemElement.textContent = "Valor máximo por carrinho: R$1.000,00";
  mensagemElement.style.opacity = "1";

  // Oculta a mensagem após 3 segundos
  setTimeout(() => {
    mensagemElement.style.opacity = "0";

    // Remove o elemento após a animação de fade out
    setTimeout(() => {
      if (mensagemElement && mensagemElement.parentNode) {
        mensagemElement.parentNode.removeChild(mensagemElement);
      }
    }, 300);
  }, 3000);
}

// Função para verificar se adicionar um produto excederá o limite
function verificarLimiteCarrinho(valorAdicional) {
  let carrinho = obterCarrinho();
  const totalAtual = carrinho.total || 0;

  return totalAtual + valorAdicional > 1000;
}

// Modificação da função adicionarAoCarrinho para capturar a imagem do OrderBump
function adicionarAoCarrinho(nomeProduto, valorProduto, cardProduto) {
  // Busca o carrinho atual
  let carrinho = obterCarrinho();

  // Captura a URL da imagem do OrderBump
  const imagemOrderBump =
    cardProduto.querySelector(".product-img").getAttribute("src") ||
    "img/obump-default.jpg";

  // Adiciona o OrderBump como item especial
  let obumpExistente = carrinho.itens.find(
    (item) => item.nome === nomeProduto && item.isOrderBump
  );

  if (obumpExistente) {
    // Aumenta a quantidade se já existir
    obumpExistente.quantidade += 1;
  } else {
    // Adiciona novo item
    carrinho.itens.push({
      id: `${Date.now()}`, // ID único para o OrderBump
      nome: nomeProduto,
      preco: valorProduto,
      quantidade: 1,
      imagem: imagemOrderBump, // Usa a imagem capturada do OrderBump
      isOrderBump: true, // Marca como OrderBump para identificação
    });
  }

  // Atualiza totais e salva
  atualizarTotaisCarrinho(carrinho);
  salvarCarrinho(carrinho);
  renderizarCarrinho();
}

// Modificação da função de remoção para manter a mesma assinatura
function removerDoCarrinho(nomeProduto, valorProduto) {
  // Busca o carrinho atual
  let carrinho = obterCarrinho();

  // Encontra o OrderBump
  const index = carrinho.itens.findIndex(
    (item) => item.nome === nomeProduto && item.isOrderBump
  );

  if (index !== -1) {
    // Se encontrou, remove
    carrinho.itens.splice(index, 1);

    // Atualiza totais e salva
    atualizarTotaisCarrinho(carrinho);
    salvarCarrinho(carrinho);
    renderizarCarrinho();
  }
}

// Modificação da função de configuração do OrderBump para passar o card do produto
function configurarOrderBump() {
  // Seleciona todos os botões de OrderBump
  const botoesOrderBump = document.querySelectorAll(".toggle-input");

  botoesOrderBump.forEach((botao) => {
    botao.addEventListener("change", function () {
      // Encontra o card do produto
      const cardProduto = this.closest(".product-card");

      // Obtém informações do produto
      const nomeProduto = cardProduto.querySelector("h3").textContent.trim();
      const valorString = cardProduto
        .querySelector(".font-bold.text-lg.text-\\[\\#59BF75\\]")
        .textContent.trim();
      const valorProduto = extrairValorNumerico(valorString);

      // Adiciona ou remove do carrinho
      if (this.checked) {
        if (verificarLimiteCarrinho(valorProduto)) {
          exibirMensagemLimite();
          // Desmarca o checkbox programaticamente
          this.checked = false;

          // Restaura a aparência visual - necessário porque o evento visual já ocorreu
          const toggleDot = cardProduto.querySelector(".toggle-dot");
          const toggleText = cardProduto.querySelector(".toggle-text");
          const purchasedBadge = cardProduto.querySelector(".purchased-badge");

          if (toggleDot) toggleDot.classList.add("hidden");
          if (toggleText) toggleText.textContent = "PEGAR OFERTA";
          if (purchasedBadge) purchasedBadge.classList.add("hidden");

          // Impede que o código continue
          return;
        }
        adicionarAoCarrinho(nomeProduto, valorProduto, cardProduto);
        console.log(
          `Adicionei ao carrinho o produto: ${nomeProduto} - ${valorString}`
        );
      } else {
        removerDoCarrinho(nomeProduto, valorProduto);
        console.log(
          `Removi do carrinho o produto: ${nomeProduto} - ${valorString}`
        );
      }
    });
  });
}

// Obtém o carrinho do localStorage
function obterCarrinho() {
  const carrinhoSalvo = localStorage.getItem("wepinkCarrinho");
  if (carrinhoSalvo) {
    return JSON.parse(carrinhoSalvo);
  } else {
    // Carrinho padrão se não existir
    return {
      itens: [],
      total: 0,
      desconto: 0,
    };
  }
}

// Salva o carrinho no localStorage
function salvarCarrinho(carrinho) {
  localStorage.setItem("wepinkCarrinho", JSON.stringify(carrinho));
}

// Calcula e atualiza os totais do carrinho
function atualizarTotaisCarrinho(carrinho) {
  // Calcula o novo total
  const total = carrinho.itens.reduce(
    (sum, item) => sum + item.preco * item.quantidade,
    0
  );

  // Garante que o total não fique negativo
  carrinho.total = Math.max(0, total);
}

// Função para renderizar o carrinho na tela
function renderizarCarrinho() {
  // Obtém o carrinho
  const carrinho = obterCarrinho();

  // Atualiza o contador
  atualizarContadorCarrinho(carrinho);

  // Renderiza os produtos no resumo da compra
  renderizarProdutosCarrinho(carrinho);

  // Atualiza os valores exibidos
  atualizarValoresExibidos(carrinho);
}

// Atualiza o contador do carrinho
function atualizarContadorCarrinho(carrinho) {
  const contador = carrinho.itens.reduce(
    (total, item) => total + item.quantidade,
    0
  );

  // Procura o elemento contador
  const contadorElement = document.getElementById("contador-carrinho");
  if (contadorElement) {
    // Atualiza o valor
    contadorElement.textContent = contador;
    // Mostrar/ocultar conforme necessário
    contadorElement.style.display = contador > 0 ? "flex" : "none";
  }
}

// Renderiza os produtos no resumo do carrinho
function renderizarProdutosCarrinho(carrinho) {
  const summaryBody = document.getElementById("purchase-summary__body");
  if (!summaryBody) return;

  // Limpa produtos existentes, mantendo apenas a estrutura base
  summaryBody.innerHTML = "";

  // Adiciona cada produto
  carrinho.itens.forEach((item) => {
    // Define classe para OrderBumps
    const classeOrderBump = item.isOrderBump
      ? "bg-[#FFF5EE] p-2 rounded-md"
      : "";

    const produtoHTML = `
  <div class="flex items-center mb-4 ${classeOrderBump}" data-id="${item.id}">
    <img class="w-16 h-16 object-cover" src="${item.imagem}" alt="${item.nome}" 
         onerror="this.src='img/obump-default.jpg'"/>
    <div class="ml-4 flex-1">
      <span class="block text-base font-medium break-words" style="word-wrap: break-word; overflow-wrap: break-word; hyphens: auto; display: -webkit-box; -webkit-line-clamp: 2; -webkit-box-orient: vertical; overflow: hidden;">
        ${item.isOrderBump ? "🎁 " : ""}${item.nome}
      </span>
      <span class="text-sm text-gray-500">R$ ${formatarMoeda(item.preco)}</span>
    </div>
    <div class="text-right">
      <span class="text-base">${item.quantidade} un.</span>
    </div>
  </div>
`;
    summaryBody.innerHTML += produtoHTML;
  });

  // Adiciona os totais e estrutura do resumo
  adicionarEstruturaTotais(summaryBody, carrinho);
}

// Adiciona a estrutura de totais no resumo
function adicionarEstruturaTotais(element, carrinho) {
  element.innerHTML += `
<div class="hidden ob-preview">
  <div class="ob-purchased-info"></div>
  <div class="ob-preview-content"></div>
</div>

<div class="border-t border-gray-200 mt-4"></div>

<div class="mb-4">
  <div class="flex justify-between mt-4">
    <div class="text-left text-sm">Subtotal</div>
    <div class="text-right text-sm">
      R$ <span class="font-medium subtotal-value">${formatarMoeda(
        carrinho.total
      )}</span>
    </div>
  </div>
  <div class="flex justify-between mt-2">
    <div class="text-left text-sm">Frete</div>
    <div class="text-right text-sm">Grátis</div>
  </div>

  <div id="div_progressive_discount" class="flex justify-between mb-2 ${
    carrinho.desconto > 0 ? "" : "hidden"
  } progressive-discount-class">
    <input type="hidden" id="progressive_discount" value="${
      carrinho.desconto
    }"/>
  </div>

  <div class="flex justify-between mb-1 hidden automatic-discount">
    <div class="text-left text-sm">Desconto cartão</div>
    <div class="text-right text-sm">
      <span class="value-automatic-discount discount-span"></span>
    </div>
  </div>
</div>

<div class="border-t border-gray-200 mt-4"></div>

<div class="relative mt-4">
  <div class="flex justify-between">
    <div class="text-left text-base">Total</div>
    <div class="text-right text-base font-semibold total-value">
      R$ ${formatarMoeda(carrinho.total)}
    </div>
  </div>
</div>
`;
}

// Atualiza os valores exibidos na tela
function atualizarValoresExibidos(carrinho) {
  // Atualiza subtotal
  const subtotalElements = document.querySelectorAll(".subtotal-value");
  subtotalElements.forEach((el) => {
    el.textContent = formatarMoeda(carrinho.total);
  });

  // Atualiza total
  const totalElements = document.querySelectorAll(".total-value");
  totalElements.forEach((el) => {
    el.textContent = `R$ ${formatarMoeda(carrinho.total)}`;
  });
}

// Formata valor para exibição em moeda brasileira
function formatarMoeda(valor) {
  return valor.toLocaleString("pt-BR", {
    minimumFractionDigits: 2,
    maximumFractionDigits: 2,
  });
}

// Verificar estado dos OrderBumps ao carregar a página
function sincronizarOrderBumpsComCarrinho() {
  const carrinho = obterCarrinho();
  const botoesOrderBump = document.querySelectorAll(".toggle-input");

  // Verifica cada OrderBump
  botoesOrderBump.forEach((botao) => {
    const cardProduto = botao.closest(".product-card");
    const nomeProduto = cardProduto.querySelector("h3").textContent.trim();

    // Verifica se já está no carrinho
    const jaNoCarrinho = carrinho.itens.some(
      (item) => item.nome === nomeProduto && item.isOrderBump
    );

    // Marca o checkbox conforme estado no carrinho
    botao.checked = jaNoCarrinho;

    // Atualiza a visualização do botão
    if (jaNoCarrinho) {
      const toggleDot = cardProduto.querySelector(".toggle-dot");
      const toggleText = cardProduto.querySelector(".toggle-text");
      const purchasedBadge = cardProduto.querySelector(".purchased-badge");

      if (toggleDot) toggleDot.classList.remove("hidden");
      if (toggleText) toggleText.textContent = "REMOVER OFERTA";
      if (purchasedBadge) purchasedBadge.classList.remove("hidden");
    }
  });
}

// Inicialização e execução da sincronização ao carregar
function inicializarCarrinho() {
  renderizarCarrinho();
  sincronizarOrderBumpsComCarrinho();
}

//SELECT PIX AND SELECT CARD
//SELECT PIX AND SELECT CARD
// Código para alternar entre métodos de pagamento (PIX e cartão)
document.addEventListener("DOMContentLoaded", function () {
  // Referencias aos elementos
  const pixPayment = document.getElementById("pix-payment-0");
  const cardPayment = document.getElementById("card-payment-0");
  const orderBumps = document.getElementById("order-bumps");
  const pixDataPayment = document.getElementById("pix_data_payment");
  const cardDataPayment = document.getElementById("card_data_payment");

  // Estilo para item selecionado
  const selectedStyle = "box-shadow: inset 0 0 0 2px #fe017b;";

  // Função para selecionar o método PIX
  function selectPix() {
    console.log("executing selectPix()");
    // Atualiza estilos
    pixPayment.setAttribute("style", selectedStyle);
    cardPayment.removeAttribute("style");

    // Mostra/oculta elementos
    orderBumps.style.display = "block";
    pixDataPayment.style.display = "block";
    cardDataPayment.style.display = "none";

    // Atualiza botões
    const finishButton = document.getElementById("finalize_pix_purchase");
    if (finishButton) finishButton.style.display = "block";

    const cardButton = document.querySelector(
      ".btn-holder-payment .buy-btn:not(#finalize_pix_purchase)"
    );
    if (cardButton) cardButton.style.display = "none";
  }

  // Função para selecionar o método cartão
  function selectCard() {
    console.log("executing selectCard()");
    // Atualiza estilos
    cardPayment.setAttribute("style", selectedStyle);
    pixPayment.removeAttribute("style");

    // Mostra/oculta elementos
    orderBumps.style.display = "none";
    pixDataPayment.style.display = "none";
    cardDataPayment.style.display = "block";

    // Atualiza botões
    const finishButton = document.getElementById("finalize_pix_purchase");
    if (finishButton) finishButton.style.display = "none";

    const cardButton = document.querySelector(
      ".btn-holder-payment .buy-btn:not(#finalize_pix_purchase)"
    );
    if (cardButton) cardButton.style.display = "block";
  }

  // Event listeners
  pixPayment.addEventListener("click", selectPix);
  cardPayment.addEventListener("click", selectCard);

  // Configura estado inicial (PIX selecionado por padrão)
  selectPix();
});
