<?php
/**
 * Classe FacebookAttributionHelper
 * 
 * Classe auxiliar para gerenciar parâmetros de atribuição do Facebook
 */
class FacebookAttributionHelper {
    /**
     * Captura o fbclid da URL e o armazena como cookie _fbc
     * 
     * @return string|null Valor do FBC formatado ou null se não disponível
     */
    public static function captureFbclid() {
        // Se o fbclid está presente na URL, captura e armazena como cookie
        if (isset($_GET['fbclid'])) {
            $fbclid = $_GET['fbclid'];
            // Gera o valor do _fbc conforme especificação do Facebook (fb.1.timestamp.fbclid)
            $fbc = 'fb.1.' . time() . '.' . $fbclid;
            // Define o cookie com validade de 90 dias (7776000 segundos)
            setcookie('_fbc', $fbc, time() + 7776000, '/', '', isset($_SERVER['HTTPS']), true);
            return $fbc;
        }
        
        // Se não está na URL, verifica se já existe no cookie
        return $_COOKIE['_fbc'] ?? null;
    }
    
    /**
     * Obtém o valor do cookie _fbp (Facebook Browser ID)
     * 
     * @return string|null Valor do FBP ou null se não disponível
     */
    public static function getFbp() {
        return $_COOKIE['_fbp'] ?? null;
    }
    
    /**
     * Preserva parâmetros UTM e fbclid em formulários
     * Adiciona campos ocultos para manter os parâmetros durante a navegação
     * 
     * @return string HTML com campos ocultos
     */
    public static function preserveParameters() {
        $html = '';
        $params_to_preserve = ['fbclid', 'utm_source', 'utm_medium', 'utm_campaign', 'utm_content', 'utm_term'];
        
        foreach ($params_to_preserve as $param) {
            if (isset($_GET[$param])) {
                $value = htmlspecialchars($_GET[$param], ENT_QUOTES, 'UTF-8');
                $html .= "<input type='hidden' name='{$param}' value='{$value}'>\n";
            }
        }
        
        return $html;
    }
    
    /**
     * Gera um ID de evento consistente para uso em eventos duplicados
     * (quando enviando o mesmo evento pelo pixel e pela API de Conversão)
     * 
     * @param string $event_name Nome do evento (ex: Purchase, AddToCart)
     * @param string $unique_identifier Um identificador único (ex: order_id, session_id)
     * @return string ID do evento formatado
     */
    public static function generateEventId($event_name, $unique_identifier) {
        return strtolower($event_name) . '_' . $unique_identifier . '_' . time();
    }
    
    /**
     * Captura todos os parâmetros UTM da URL atual
     * 
     * @return array Array associativo com parâmetros UTM
     */
    public static function captureUtmParameters() {
        $utm_params = [];
        $possible_params = [
            'utm_source', 'utm_medium', 'utm_campaign', 
            'utm_term', 'utm_content', 'fbclid'
        ];
        
        foreach ($possible_params as $param) {
            if (isset($_GET[$param])) {
                $utm_params[$param] = $_GET[$param];
            }
        }
        
        return $utm_params;
    }
    
    /**
     * Cria um cookie para armazenar parâmetros UTM
     * Útil para atribuição em visitas posteriores
     * 
     * @param array $utm_params Parâmetros UTM a serem armazenados
     * @param int $expiry Validade do cookie em segundos (padrão: 30 dias)
     */
    public static function storeUtmParameters($utm_params = null, $expiry = 2592000) {
        if ($utm_params === null) {
            $utm_params = self::captureUtmParameters();
        }
        
        if (!empty($utm_params)) {
            setcookie(
                'utm_data', 
                json_encode($utm_params), 
                time() + $expiry, 
                '/', 
                '', 
                isset($_SERVER['HTTPS']), 
                true
            );
        }
    }
    
    /**
     * Recupera parâmetros UTM armazenados em cookies
     * 
     * @return array Parâmetros UTM armazenados
     */
    public static function getStoredUtmParameters() {
        if (isset($_COOKIE['utm_data'])) {
            return json_decode($_COOKIE['utm_data'], true) ?: [];
        }
        return [];
    }
    
    /**
     * Combina os parâmetros UTM atuais com os armazenados
     * Prioriza os parâmetros da visita atual
     * 
     * @return array Parâmetros UTM combinados
     */
    public static function getMergedUtmParameters() {
        $stored_params = self::getStoredUtmParameters();
        $current_params = self::captureUtmParameters();
        
        return array_merge($stored_params, $current_params);
    }
    
    /**
     * Adiciona parâmetros UTM combinados aos dados do evento
     * 
     * @param array $event_data Dados do evento a serem enriquecidos
     * @return array Dados do evento com parâmetros UTM
     */
    public static function addUtmParametersToEvent($event_data) {
        $utm_params = self::getMergedUtmParameters();
        
        if (!empty($utm_params)) {
            if (!isset($event_data['custom_data'])) {
                $event_data['custom_data'] = [];
            }
            
            // Adiciona parâmetros UTM aos dados personalizados
            foreach ($utm_params as $key => $value) {
                $event_data['custom_data']['utm_' . str_replace('utm_', '', $key)] = $value;
            }
        }
        
        return $event_data;
    }
}